/***************************************************************************
 *            errorcalc.cpp
 *
 *  Copyright (c) 2003 Lukasz Tomicki <tomicki@o2.pl> www.tomicki.net
 *  
 ****************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "errorcalc.h"

using namespace std;

CErrorCalc::CErrorCalc(float **p, UINT size) : 
discarded(0), values(0), _average(0), _delta(0), _std(0)
{
	data = new list<float*>;
	for (int i(0); i < size; ++i) {
		float *f = new float;
		memcpy(f, &p[i], sizeof(float));
		data->push_back(f);
	}
}

CErrorCalc::CErrorCalc(const char *file) : 
discarded(0), values(0), _average(0), _delta(0), _std(0)
{
	ReadData(file);
}

CErrorCalc::CErrorCalc(FILE *hFile) : 
discarded(0), values(0), _average(0), _delta(0), _std(0)
{
	ReadData(hFile);
}

CErrorCalc::~CErrorCalc()
{
	CleanUp();
}

void CErrorCalc::SetOptions(ErrorCalcOptions _options)
{
	options = _options;
	return;
}

void CErrorCalc::ReadData(FILE *hFile)
{
	data = new list<float*>;

	if (hFile) {
		fseek(hFile, 0, SEEK_SET);
		char m_number[100];
		memset(m_number, 0, sizeof(m_number));
		UINT m_CurrentChar(0);
		char c;
		float *m_current;

		do {
			c = fgetc(hFile);
			if (c == '\n' || c == '\r' || c == ' ' || c == EOF || 
					m_CurrentChar == 100) {
				if (m_CurrentChar != 0) {
					m_current = new float;
					sscanf(m_number, "%f", m_current);
					data->push_back(m_current);
					memset(m_number, 0, sizeof(m_number));
					m_CurrentChar = 0;
				}
			} else {
				m_number[m_CurrentChar] = c;
				++m_CurrentChar;
			}
		} while (c != EOF);

		fclose(hFile);
	}

	return;
}

void CErrorCalc::ReadData(const char *file)
{
	data = new list<float*>;

	if (file) {
		char m_number[100];
		memset(m_number, 0, sizeof(m_number));
		UINT m_CurrentChar(0);
		char c;
		const char *filepos = file;
		float *m_current = new float;

		do {
			c = *filepos;
			++filepos;
			if (c == ' ' || c == 0 || m_CurrentChar == 100) {
				if (m_CurrentChar != 0) {
					sscanf(m_number, "%f", m_current);
					data->push_back(m_current);
					m_current = new float;
					memset(m_number, 0, sizeof(m_number));
					m_CurrentChar = 0;
				}
			} else {
				m_number[m_CurrentChar] = c;
				++m_CurrentChar;
			}
		} while (c != 0);
	}

	return;
}

void CErrorCalc::RemoveValuesOutOfRange()
{
	list<float*>::iterator i = data->begin();
	float Xmin(**i);
	float Xmax(**i);
	for (; i != data->end(); ++i) {
		if (**i < Xmin)
			Xmin = **i;

		if (**i > Xmax)
			Xmax = **i;
	}

	//printf("Found Xmax = %f\n", Xmax);
	//printf("Found Xmin = %f\n", Xmin);

	float Xdelta((Xmax - Xmin) / 2);
	//printf("Found Xdelta = %f\n", Xdelta);
	for (i = data->begin(); i != data->end(); ++i) {
		if (**i >= average + Xdelta || **i <= average - Xdelta) {
			delete *i;
			data->erase(i);
			i = data->begin();
			++discarded;
		}
	}
}

void CErrorCalc::PrintVerboseInfo()
{
	for (list<float*>::iterator i = data->begin(); i != data->end(); ++i)
		printf("%f\n", **i);
}

float inline CErrorCalc::MakeUnsigned(float input)
{ 
	return ( input > 0 ? input : -input );
}

void CErrorCalc::CountAvg()
{
	float sum(0);
	for (list<float*>::iterator i = data->begin(); i != data->end(); ++i)
		sum += **i;
	
	UINT num(data->size());
	
	values = num;
	average = sum / num;
	_average = 1;
	
	if (options.discard)
		RemoveValuesOutOfRange();
	
	return;
}

void CErrorCalc::CountDeltaAverage()
{
	if (!_average)
		CountAvg();
	
	float sum(0);
	for (list<float*>::iterator i = data->begin(); i != data->end(); ++i)
		sum += MakeUnsigned(average - **i);
	
	deltaError  = sum / data->size();
	_delta = 1;
	return;
}

void CErrorCalc::CountDeltaSqr()
{
	if (!_average)
		CountAvg();
	
	float sum(0);
	for (list<float*>::iterator i = data->begin(); i != data->end(); ++i)
		sum += pow(**i - average, 2);
	
	sum /= data->size();
	stdError = sqrt(sum) ;
	_std = 1;
	return;
}

void CErrorCalc::eval()
{
	CountAvg();
	CountDeltaAverage();
	CountDeltaSqr();
}

void CErrorCalc::CleanUp()
{
	for (list<float*>::iterator i = data->begin(); i != data->end(); ++i)
		delete *i;
	
	delete data;
}

UINT CErrorCalc::getDiscarded()
{
	if (!_average)
		CountAvg();
	
	return ( discarded );
}

UINT CErrorCalc::getTotal()
{
	if (!_average)
		CountAvg();
	
	return ( values );
}

float CErrorCalc::getAverage()
{
	if (!_average)
		CountAvg();

	return ( average );
}

float CErrorCalc::getErrorDelta()
{
	if (!_delta)
		CountDeltaAverage();
	
	return ( deltaError );
}

float CErrorCalc::getErrorStd()
{
	if (!_std)
		CountDeltaSqr();
	
	return ( stdError );
}
