/***************************************************************************
 *  chimera.cc : This file is part of 'chimera'
 *
 *  (c) 2003,2004 by Lukasz Tomicki <tomicki@o2.pl>
 *
 ****************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "chimera.h"

using namespace std;

FILE *hFile = 0;
const char *passwd = 0;
CServiceAttack *crack = 0;

const char* get_single_passwd(bool r)
{
	static bool t(true);
	
	if (r) {
		t = true;
		return 0;
	}
	
	if (t) {
		t = false;
		return ( passwd );
	} else
		return 0;
}

/* rev 1.1.4 */

const char* get_passwd(bool r)
{
	if (r) {
		rewind(hFile);
		return 0;
	}
	
	const char *p = CServiceAttack::get_next_line(hFile);
	
	if (!p)
		rewind(hFile);
	
	return ( p );
}

void print_data(const char *p)
{
	printf("%s", p);
//	fflush(stdout);
}

void deal_with_results(CServiceAttack::target_output_data *p)
{
	char *ptr = 0;
	printf("service: ");
	switch (p->type) {
		case CServiceAttack::SERVICE_POP3:
			printf("pop3");
			printf(" host: %s port: %d user: %s password: %s\n", 
				inet_ntoa(p->addr.sin_addr), ntohs(p->addr.sin_port), p->user, 
				p->passwd ? p->passwd : "(null)");
		break;
			
		case CServiceAttack::SERVICE_FTP:
			printf("ftp");
			printf(" host: %s port: %d user: %s password: %s\n", 
				inet_ntoa(p->addr.sin_addr), ntohs(p->addr.sin_port), p->user, 
				p->passwd ? p->passwd : "(null)");
		break;
			
		case CServiceAttack::SERVICE_GG:
			printf("gg");
			printf(" host: %s port: %d user: %s password: %s\n", 
				inet_ntoa(p->addr.sin_addr), ntohs(p->addr.sin_port), p->user, 
				p->passwd ? p->passwd : "(null)");
		break;
			
		case CServiceAttack::SERVICE_VNC:
			printf("vnc");
			printf(" host: %s port: %d user: %s password: %s\n", 
				inet_ntoa(p->addr.sin_addr), ntohs(p->addr.sin_port), p->user, 
				p->passwd ? p->passwd : "(null)");
		break;
			
		case CServiceAttack::SERVICE_MYSQL:
			printf("mysql");
			printf(" host: %s port: %d user: %s password: %s\n", 
				inet_ntoa(p->addr.sin_addr), ntohs(p->addr.sin_port), p->user, 
				p->passwd ? p->passwd : "(null)");
		break;
			
		case CServiceAttack::SERVICE_HTTP:
			printf("http");
			if (p->misc) {
				ptr = p->misc + 4;
			}
			printf(" host: %s request: %s port: %d user: %s password: %s\n", 
				inet_ntoa(p->addr.sin_addr), ptr, ntohs(p->addr.sin_port), p->user, 
				p->passwd ? p->passwd : "(null)");
		break;
	}
	
	fflush(stdout);
	
	delete p;
}

bool parse_command_line_args(CServiceAttack *p, int argc, char **argv)
{
	CServiceAttack::target_input_data target_host;
	char *host = 0;
	char *host_file = 0;
	char *user_file = 0;
	bool ret(true);
	u16 port(0);
	uint mlevel(CServiceAttack::MSG_ERROR);
	FILE *hHost = 0;
	bool bHost = true;
	char *misc = 0;
	
	char o;
	while ((o = getopt(argc, argv, "s:u:U:p:P:c:C:m:M:h:H:e:lqvt:T:nyrab:B:")) != -1) {
		switch (o) {
			case 'b':
				p->set_thread_creation_speed(atoi(optarg));
			break;
			
			case 'a':
				p->set_thread_reduction(false);
			break;
			
			case 's':
				if (!strcmp(optarg, "pop3"))
					target_host.type = CServiceAttack::SERVICE_POP3;
				
				else if (!strcmp(optarg, "ftp"))
					target_host.type = CServiceAttack::SERVICE_FTP;
				
				else if (!strcmp(optarg, "vnc"))
					target_host.type = CServiceAttack::SERVICE_VNC;
				
				else if (!strcmp(optarg, "mysql"))
					target_host.type = CServiceAttack::SERVICE_MYSQL;
				
				else if (!strcmp(optarg, "http"))
					target_host.type = CServiceAttack::SERVICE_HTTP;
			break;
				
			case 'u':
				target_host.user = alloc_copy(optarg);
			break;
			
			case 'U':
				user_file = alloc_copy(optarg);
			break;
			
			case 'p':
				passwd = alloc_copy(optarg);
			break;
			
			case 'P':
				if (!strcmp(optarg, "stdin"))
					hFile = stdin;
				else {
					hFile = fopen(optarg, "r");
					if (!hFile) {								
						printf("***unable to open password file\n");
						perror("***fopen");
						ret = false;
						goto clean_up;
					}
				}
			break;
			
			case 'c':
				switch (optarg[0]) {
					case '1':
						target_host.charset = "abcdefghijklmnopqrstuvwyxz";
					break;
					
					case '2':
						target_host.charset = "abcdefghijklmnopqrstuvwyxzABCDEFGHIJKLMNOPQRSTUVWYXZ";
					break;
					
					case '3':
						target_host.charset = "abcdefghijklmnopqrstuvwyxzABCDEFGHIJKLMNOPQRSTUVWYXZ1234567890";
					break;
					
					case '4':
						target_host.charset = "abcdefghijklmnopqrstuvwyxzABCDEFGHIJKLMNOPQRSTUVWYXZ1234567890!@#$%^&*()_-+=]}[{'\";:/?.>,<\\|";
					break;
				}
			break;
			
			case 'C':
				target_host.charset = alloc_copy(optarg);
			break;
			
			case 'm':
				target_host.min_chars = max(0, min(30, atoi(optarg)));
			break;
			
			case 'M':
				target_host.max_chars = max(0, min(30, atoi(optarg)));
			break;
			
			case 'h':
				host = alloc_copy(optarg);
			break;
			
			case 'H':
				host_file = alloc_copy(optarg);
			break;
			
			case 'e':
				port = max(0, atoi(optarg));
			break;
			
			case 'l':
#ifdef OPENSSL
				target_host.ssl = true;
#else
				printf("***to use SSL you, please compile chimera with OpenSSL support\n");
#endif
			break;
			
			case 'q':
				mlevel = CServiceAttack::MSG_NONE;
			break;
			
			case 'v':
				mlevel = CServiceAttack::MSG_VERBOSE;
			break;
			
			case 't':
				p->set_threads(max(0, min(16384, atoi(optarg))));
			break;
			
			case 'T':
				p->set_conn_timeout(atoi(optarg));
			break;
			
			case 'B':
				p->set_conn_errors_max(atoi(optarg));
			break;
			
			case 'n':
				target_host.passwd_options = 
			CServiceAttack::PasswdOptions(target_host.passwd_options | 
			CServiceAttack::PASSWD_OPTIONS_NULL);
			break;
			
			case 'y':
				target_host.passwd_options = 
			CServiceAttack::PasswdOptions(target_host.passwd_options | 
			CServiceAttack::PASSWD_OPTIONS_LOGIN);
			break;
			
			case 'r':
				target_host.passwd_options = 
			CServiceAttack::PasswdOptions(target_host.passwd_options | 
			CServiceAttack::PASSWD_OPTIONS_LOGIN_REV);
			break;
		}
	}
	p->set_output_level((CServiceAttack::MessageLevel) mlevel);
	p->set_output(print_data);
	p->set_notice(deal_with_results);
	
	if (!port) {
		switch (target_host.type) {
			case CServiceAttack::SERVICE_FTP:
				port = 21;
			break;
			
			case CServiceAttack::SERVICE_MYSQL:
				port = 3306;
			break;
			
			case CServiceAttack::SERVICE_POP3:
				if (!target_host.ssl)
					port = 110;
				else
					port = 995;
			break;
			
			case CServiceAttack::SERVICE_HTTP:
				if (!target_host.ssl)
					port = 80;
				else
					port = 443;
			break;
			
			case CServiceAttack::SERVICE_VNC:
			break;
		}
	}
	
	if (passwd)
		p->set_passwd_func(get_single_passwd);
	else if (hFile) {
		p->set_passwd_func(get_passwd);
	} else if (target_host.charset) {
		if (!target_host.min_chars)
			target_host.min_chars = 1;
		if (!target_host.max_chars)
			target_host.max_chars = 8;
		if (target_host.max_chars < target_host.min_chars) {
			printf("***incorrect value of max-chars. max-chars must be >= min-chars\n");
			ret = false;
			goto clean_up;
		}
	} else if (!(target_host.passwd_options & CServiceAttack::PASSWD_OPTIONS_NULL) &&
			!(target_host.passwd_options & CServiceAttack::PASSWD_OPTIONS_LOGIN) &&
			!(target_host.passwd_options & CServiceAttack::PASSWD_OPTIONS_LOGIN_REV)) {
		printf("***no password source set. Use at least one of the below.\n");
		printf("   a single password (-p password)\n");
		printf("   a password file (-P password file)\n");
		printf("   stdin (-P stdin)\n");
		printf("   a pre-defined brute force charset (-c 1-4)\n");
		printf("   a user-defined brute force charset (-C charset)\n");
		printf("   null passwords (-n)\n");
		printf("   passwords same as login (-y)\n");
		printf("   passwords same as reversed login (-r)\n");
		ret = false;
		goto clean_up;
	}
	
	if (target_host.type == CServiceAttack::SERVICE_NONE) {
		printf("***no service selected - use the syntax '-s service' to select one\n");
		ret = false;
		goto clean_up;
	}
	
	if (target_host.type == CServiceAttack::SERVICE_HTTP) {
		char *ptr, *ptr2;
		
		if (ptr2 = strstr(host , "://")) {
			ptr2 += 3;
			
			if (ptr = index(ptr2, '/')) {
				u8 extra(0);
				if (*(ptr + strlen(ptr) - 1) != '/')
					extra = 1;
				
				u32 request_size = strlen(host) + extra + 1 /* null term */;
				u32 remote_path_size = (ptr - ptr2) + 1 /* null term */;
				u32 misc_size = 4 /* 4 byte request size */ + 
					request_size /* request */ +
					4 /* 4 byte remote path size */ +
					remote_path_size /* remote path */;
				
				misc = new char[misc_size];
				
				*misc = request_size;
				memcpy(misc + 4, host, request_size - extra - 1);
				*(misc + request_size + 4) = remote_path_size;
				memcpy(misc + 8 + request_size, ptr2, remote_path_size - extra - 1);
				
				*(misc + 4 + request_size - 1) = 0;
				*(misc + 8 + request_size + remote_path_size - 1) = 0;
				
				if (extra) {
					*(misc + 4 + request_size - 2) = '/';
				}
				
				target_host.misc_size = misc_size;
				target_host.misc = misc;
				
				*ptr = 0;
				ptr = host;
				host = alloc_copy(ptr2);
				delete [] ptr;
			} else {
				printf("***bad hostname supplied. When using http cracking, use:\n");
				printf("   http://hostname/remote_server_path\n");	
				ret = false;
				goto clean_up;
			}
	  	} else {
			printf("***bad hostname supplied. When using http cracking, the host syntax is:\n");
			printf("   http://hostname/remote_server_path\n");	
			ret = false;
			goto clean_up;
		}
	}
	
	while (bHost) {
		if (host && port) {
			void *host_data = CServiceAttack::get_host_data(host, port);
			if (!host_data) {
				printf("***unable to resolve hostname\n");
				ret = false;
				goto clean_up;
			}
			
			memcpy(&target_host.addr, host_data,
			sizeof(target_host.addr));
			bHost = false;
		} else if (host_file) {
			hHost = fopen(host_file, "r");	
			if (!hHost) {
				printf("***unable to open hostname file\n");
				perror("***fopen");
				ret = false;
				goto clean_up;
			}
		} else {
			printf("***no hostname supplied\n");
			bHost = false;
			continue;
		}
		
		FILE *hUser = 0;
		if (user_file) {
			hUser = fopen(user_file, "r");
			if (!hUser) {
				printf("***unable to open username file\n");
				perror("***fopen");
				ret = false;
				goto clean_up;
			}
			
			if (hUser) {
				const char *user_name = 0;
				while (user_name = alloc_copy(CServiceAttack::get_next_line(hUser))) {
					target_host.user = user_name;
					p->add_target(target_host);
				}		
				fclose(hUser);
			}
		} else {
			if (!target_host.user) {
				printf("***no username set\n");
				ret = false;
				goto clean_up;
			}
			p->add_target(target_host);
		}
	}

clean_up:
	
	if (!host)
		delete [] host;
	if (!host_file)
		delete [] host_file;
	if (!user_file)
		delete [] target_host.user;
	if (!target_host.charset)
		delete [] target_host.charset;
	if (!misc)
		delete [] misc;
	
	return ( ret );
}

void clean_up()
{
	if (passwd)
		delete [] passwd;
	if (hFile && hFile != stdin)
		fclose(hFile);
	if (crack)
		delete crack;
}

void print_usage()
{
	puts("chimera - remote passwd cracking tool v0.28.2");
	puts("(c) 2003,2004 by Lukasz Tomicki <tomicki@o2.pl>");
	puts(" usage: chimera [-s service] [-h host/IP] [-u/-U user/userlist]");
	puts("   [-p/-P/-c/-C password/passwordlist/pre-defined charset/user charset] <options>\n");
	puts(" options:");
	puts("  -s sets the service you want to attack to the given type");
	puts("     valid services are: ftp, pop3, mysql");
	puts("  -h host to attack");
	puts("  -H sets a file of hosts to attack");
	puts("  -e port to attack on the remote host (if not standard)");
	puts("  -u sets the user name for the service we are attacking");
	puts("  -U sets a file of user names to use");
	puts("  -p sets a passwd to check");
	puts("  -P sets a file of passwds to check (set 'stdin' to feed from");
	puts("     standard input");
	
	puts("  -n check for null passwords");
	puts("  -y check for passwodrs same as the user name");
	puts("  -r check for passwords same as the reversed user name");
	
	puts("  -c <n> use pre-defined charset");
	puts("      1 abcdefghijklmnopqrstuvwyxz");
	puts("      2 abcdefghijklmnopqrstuvwyxzABCDEFGHIJKLMNOPQRSTUVWYXZ");
	puts("      3 abcdefghijklmnopqrstuvwyxzABCDEFGHIJKLMNOPQRSTUVWYXZ1234567890");
	puts("      4 abcdefghijklmnopqrstuvwyxzABCDEFGHIJKLMNOPQRSTUVWYXZ1234567890!@#$%^&*()_-+=]}[{'\";:/?.>,<\\|");
	puts("\n  -C sets a user defined brute force charset");
	puts("    example: -C qwerty1234567890\n");
	puts("  -m mininum number of characters to use (default: 1)");
	puts("  -M maximum number of characters to use (default: 8)");
	
	puts("  -l use ssl ( default: no )");
	puts("  -t number of threads to use (default: 64)");
	puts("  -T set connection timeout (default: 60)");
	puts("  -B set max connection errors (default: 5)");
	puts("  -b number of microseconds between creating new threads (default: 250)");
	puts("  -a do not reduce thread number on connection errors (experts only)");
	puts("  -q quiet mode - output only successfully cracked accounts");
	puts("  -v verbose mode - output every password attempt");
}

int main(int argc, char **argv)
{	
	if (argc == 1) {
		print_usage();
		return 0;
	}
	
	crack = new CServiceAttack();
	
	if (parse_command_line_args(crack, argc,  argv))
		crack->run_atack();
	
	clean_up();
	
	return (0);
}
