/***************************************************************************
 *  CServiceAttack.cc : This file is part of 'chimera'
 *
 *  (c) 2003,2004 by Lukasz Tomicki <tomicki@o2.pl>
 *	Documentation @ http://tomicki.net/
 *
 ****************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "CServiceAttack.h"

using namespace std;

#ifdef OPENSSL
	RSA *CServiceAttack::rsa = 0;
#endif

CServiceAttack::CServiceAttack() : threads_running(0), threads(autobalance), 
output_level(MSG_NONE), thread_reduce(true), thread_creation_speed(2500), std_timeout(60), 
max_connection_errors(5)
{
	write_output = 0;
	next_passwd = 0;
	notice_func = 0;

#ifdef WINDOWS

	WORD wVersionRequested;
	WSADATA wsaData;
 
	wVersionRequested = MAKEWORD(2, 2);
	 
	if (WSAStartup(wVersionRequested, &wsaData)) {
		printf("***unable to initialize WinSock\n"); 
		return 0;
	}

	if (LOBYTE(wsaData.wVersion) != 2 || HIBYTE(wsaData.wVersion) != 2) {
		WSACleanup();
		printf("***unable to initialize WinSock\n"); 
	}
#endif
	
}

CServiceAttack::~CServiceAttack()
{
	flush();
}

void CServiceAttack::flush()
{
	while (is_working())
		pthread_cond_signal(&thread_wait);
	
	for (cracking_dataIter i = cracking_data.begin(); i != cracking_data.end(); ++i)
		delete *i;
	
	for (cracking_dataIter i = old_data.begin(); i != old_data.end(); ++i)
		delete *i;
}

void CServiceAttack::set_thread_creation_speed(uint s)
{
	thread_creation_speed = max((uint)0, s);
}

uint CServiceAttack::get_thread_creation_speed()
{
	return ( thread_creation_speed );
}

void CServiceAttack::set_thread_reduction(bool r)
{
	thread_reduce = r;
}

bool CServiceAttack::get_thread_reduction()
{
	return ( thread_reduce );
}

void CServiceAttack::set_notice(void (*p)(target_output_data*))
{
	notice_func = p;
}

void CServiceAttack::set_output(void (*p)(const char*))
{
	write_output = p;
}

void CServiceAttack::set_passwd_func(const char* (*p)(bool))
{
	next_passwd = p;
}

void CServiceAttack::set_output_level(MessageLevel level)
{
	output_level = level;
}

uint CServiceAttack::get_conn_timeout()
{
	return ( std_timeout );
}

void CServiceAttack::set_conn_timeout(uint t)
{
	std_timeout = t;
}

uint CServiceAttack::get_conn_errors_max()
{
	return ( max_connection_errors );
}

void CServiceAttack::set_conn_errors_max(uint t)
{
	max_connection_errors = t;
}

CServiceAttack::MessageLevel CServiceAttack::get_output_level()
{
	return ( (MessageLevel) output_level );
}

void CServiceAttack::set_threads(uint n)
{
	threads = max((uint)0, n);
}

uint CServiceAttack::get_threads()
{
	return ( threads );
}

void CServiceAttack::add_target(target_input_data &data, bool auto_start)
{
	if (!data.type)
		return;

	thread_data *target = new thread_data(data);
	cracking_data.push_back(target);
	if (auto_start && !is_working())
		run_atack();
}

const char* CServiceAttack::rev_string(const char *p)
{
	static char buffer[256];
	uint size(strlen(p));
	if (size >= 256)
		return 0;
	
	for (uint i(0); i < size; ++i)
		buffer[i] = p[size - i - 1];
	
	return ( buffer );
}

const char* CServiceAttack::get_next_passwd(thread_data *p, char *c, u32 s)
{
	static const char *str = 0;
	
	pthread_mutex_t password_mutex = PTHREAD_MUTEX_INITIALIZER;
	pthread_mutex_lock(&password_mutex);
	
	get_next_passwd_begin:
	switch (p->passwd_state) {
		case PASSWD_STATE_START:
			if (p->passwd_options & PASSWD_OPTIONS_NULL) {
				p->passwd_state = PASSWD_STATE_TRY_NULL;
			} else if (p->passwd_options & PASSWD_OPTIONS_LOGIN) {
				p->passwd_state = PASSWD_STATE_TRY_LOGIN;
			} else if (p->passwd_options & PASSWD_OPTIONS_LOGIN_REV) {
				p->passwd_state = PASSWD_STATE_TRY_LOGIN_REV;
			} else if (next_passwd) {
				p->passwd_state = PASSWD_STATE_EXTERN;
			} else if (p->charset && (p->min_chars <= p->max_chars)) {
				p->passwd_state = PASSWD_STATE_INTERN;
			} else
				p->passwd_state = PASSWD_STATE_END;
		goto get_next_passwd_begin;
			
		case PASSWD_STATE_TRY_NULL:
			if (p->passwd_options & PASSWD_OPTIONS_LOGIN) {
				p->passwd_state = PASSWD_STATE_TRY_LOGIN;
			} else if (p->passwd_options & PASSWD_OPTIONS_LOGIN_REV) {
				p->passwd_state = PASSWD_STATE_TRY_LOGIN_REV;
			} else if (next_passwd) {
				p->passwd_state = PASSWD_STATE_EXTERN;
			} else if (p->charset && p->min_chars && p->max_chars) {
				p->passwd_state = PASSWD_STATE_INTERN;
			} else
				p->passwd_state = PASSWD_STATE_END;
			str = "";
			strncpy(c, str, s);
			pthread_mutex_unlock(&password_mutex);
		return ( str );
		
		case PASSWD_STATE_TRY_LOGIN:
			if (p->passwd_options & PASSWD_OPTIONS_LOGIN_REV) {
				p->passwd_state = PASSWD_STATE_TRY_LOGIN_REV;
			} else if (next_passwd) {
				p->passwd_state = PASSWD_STATE_EXTERN;
			} else if (p->charset && p->min_chars && p->max_chars) {
				p->passwd_state = PASSWD_STATE_INTERN;
			} else
				p->passwd_state = PASSWD_STATE_END;
			
			strncpy(c, p->user, s);
			pthread_mutex_unlock(&password_mutex);
		return ( p->user );
		
		case PASSWD_STATE_TRY_LOGIN_REV:
			if (next_passwd) {
				p->passwd_state = PASSWD_STATE_EXTERN;
			} else if (p->charset && p->min_chars && p->max_chars) {
				p->passwd_state = PASSWD_STATE_INTERN;
			} else
				p->passwd_state = PASSWD_STATE_END;
			
			str = rev_string(p->user);
			strncpy(c, str, s);
			pthread_mutex_unlock(&password_mutex);
		return ( str );
		
		case PASSWD_STATE_EXTERN:
			str = (*next_passwd)(false);
			if (str) {
				strncpy(c, str, s);
				pthread_mutex_unlock(&password_mutex);
				return ( str );
			} else if (p->charset && p->min_chars && p->max_chars) {
				p->passwd_state = PASSWD_STATE_INTERN;
			} else
				p->passwd_state = PASSWD_STATE_END;
			
			(*next_passwd)(true);
			
		goto get_next_passwd_begin;
		
		case PASSWD_STATE_INTERN:
			if (p->brute_passwd) {
				str = p->brute_passwd->getNextPasswd();
				if (str) {
					strncpy(c, str, s);
					pthread_mutex_unlock(&password_mutex);
					return ( str );
				} else
					p->passwd_state = PASSWD_STATE_END;
			} else {
				p->brute_passwd = new CBrutePasswd(p->charset);
				p->brute_passwd->rewindPasswds(p->min_chars, p->max_chars);
				output(MSG_BASIC, "***initializing BruteForce password generator\n***will generate a total of %.0f passwords\n", 
					p->brute_passwd->getPasswdNum());
			}
		goto get_next_passwd_begin;
			
		case PASSWD_STATE_END:
			break;
	}
	
	pthread_mutex_unlock(&password_mutex);
	return 0;
}

bool CServiceAttack::is_working()
{
	return ( (bool)threads_running );
}

void CServiceAttack::account_cracked(thread_data *pData, const char *p)
{
	if (notice_func) {
		target_output_data *out_data = new target_output_data(*pData, p);
		(*notice_func)(out_data);
	}
	pData->state = DONE;
}

void CServiceAttack::output(MessageLevel level, char *msg, ...)
{	
	if (!write_output)
		return;
	
	if (level > output_level)
		return;

	va_list arglist;
	char buffer[1024];

	memset(buffer, 0, sizeof(buffer));

	va_start(arglist, msg);
	vsprintf(buffer, msg, arglist);
	va_end(arglist);

	(*write_output)(buffer);
}

void CServiceAttack::adjust_connections(thread_data *pData)
{
	++pData->connection_errors;
	if (!threads)
		return;
	
	if (thread_reduce) {
		--threads;
		output(MSG_WARNING, "***reducing thread num to: %u\n", threads);
	}
	if (!threads) {
		pData->state = DONE;
		//output(MSG_WARNING, "***thread num reduced to zero\n");
	}
}

#ifdef OPENSSL
RSA *CServiceAttack::ssl_temp_rsa_cb(SSL * ssl, int e, int keylength)
{
	if (!rsa)
		rsa = RSA_generate_key(512, RSA_F4, 0, 0);
	return rsa;
}
#endif

SOCKET CServiceAttack::create_connect(sockaddr_in6 *addr, bool ssl, int domain, 
	int type, int protocol)
{
	SOCKET sock = socket(domain, type, protocol);

	if (sock < 0)
		return 0;
	
	fcntl(sock, F_SETFL, O_NONBLOCK);

	int r;
	if (domain == AF_INET) {
		r = connect(sock, (sockaddr *) addr, sizeof(sockaddr_in));
		if (r == -1 && errno != EINPROGRESS)
			return 0;
	}
		
	if (domain == AF_INET6) {
		r = connect(sock, (sockaddr *) addr, sizeof(sockaddr_in6));	
		if (r == -1 && errno != EINPROGRESS)
			return 0;
	}
	
	timeval tv;
	tv.tv_sec = std_timeout;
	tv.tv_usec = 0;

	fd_set writefds;
	FD_ZERO(&writefds);
	FD_SET(sock, &writefds);
	select(sock + 1, 0, &writefds, 0, &tv);
	if (!FD_ISSET(sock, &writefds)) /* connection timed out */
		return 0;
	
	char error[10];
	socklen_t error_size = 10;
	getsockopt(sock, SOL_SOCKET, SO_ERROR, &error, &error_size);
	
	if ((int)*error) /* error while connecting */
		return 0;
	
	if (!ssl)
		return (sock);

#ifdef OPENSSL
	else {
		SSL *pSSL = 0;
		SSL_CTX *sslContext = 0;

		SSL_load_error_strings();
		SSLeay_add_ssl_algorithms();

		if (!(sslContext = SSL_CTX_new(SSLv23_method())))
			return 0;

		SSL_CTX_set_options(sslContext, SSL_OP_ALL);
		SSL_CTX_set_default_verify_paths(sslContext);
		SSL_CTX_set_tmp_rsa_callback(sslContext, ssl_temp_rsa_cb);
		SSL_CTX_set_verify(sslContext, SSL_VERIFY_NONE, 0);

		if ((pSSL = SSL_new(sslContext)) == 0)
			return 0;

		SSL_set_fd(pSSL, sock);

		if (SSL_connect(pSSL) <= 0)
			return 0;

		return ((SOCKET) pSSL);
	}
#endif
	return (sock);
}

void *CServiceAttack::service_wrapper(void *data)
{
	CServiceAttack *ptr = (CServiceAttack *) data;

	ptr->service_atack();
	return 0;
}

void CServiceAttack::service_atack()
{
	while (threads_running <= threads && cracking_data.size()) {
		pthread_mutex_t data_mutex = PTHREAD_MUTEX_INITIALIZER;
		pthread_mutex_lock(&data_mutex);
		cracking_dataIter i;
		i = cracking_data.begin();
		thread_data *pData = *i;

		if (pData->state >= DONE) {
			pData->state = FINISHED;
			old_data.push_back(pData);
			cracking_data.erase(i);
			pthread_mutex_unlock(&data_mutex);
			pData = 0;
			continue;
		}
		pthread_mutex_unlock(&data_mutex);

		if (pData->state == WORKING) {
			switch (pData->type) {
			case SERVICE_POP3:
				service_pop3(pData);
				break;

			case SERVICE_FTP:
				service_ftp(pData);
				break;

			case SERVICE_VNC:
				service_vnc(pData);
				break;
			
			case SERVICE_SSH:
				service_ssh(pData);
				break;
			
			case SERVICE_NNTP:
				service_nntp(pData);
				break;

			case SERVICE_MYSQL:
				service_mysql(pData);
				break;

			case SERVICE_HTTP:
				service_http(pData);
				break;

			case SERVICE_HTTP_PROXY:
				service_http_proxy(pData);
				break;

			case SERVICE_IMAP:
				service_imap(pData);
				break;

			case SERVICE_LDAP:
				service_ldap(pData);
				break;

			case SERVICE_TELNET:
				service_telnet(pData);
				break;

			case SERVICE_GG:
				service_gg(pData);
				break;

			default:
				pData->state = DONE;
				break;
			}
		}
	}
	fflush(stdout);
	--threads_running;
	if (!threads_running)
		pthread_cond_signal(&thread_wait);
	
	pthread_exit(0);
}

void CServiceAttack::run_atack()
{
	if (is_working())
		return;
	
	while (threads_running < threads && cracking_data.size()) {
		pthread_t thread;
		++threads_running;
		if (pthread_create(&thread, 0, service_wrapper, this))
			--threads_running;
		
		usleep(thread_creation_speed);
	}
	
	if (threads_running) {
		pthread_cond_init(&thread_wait, 0);
		pthread_mutex_t wait_mutex = PTHREAD_MUTEX_INITIALIZER;
		pthread_mutex_lock(&wait_mutex);
		pthread_cond_wait(&thread_wait, &wait_mutex);
		pthread_mutex_unlock(&wait_mutex);
	}
			
	output(MSG_WARNING, "***no more active threads\n");
	output(MSG_WARNING, "***quiting...\n");
}

sockaddr_in6 *CServiceAttack::get_host_data(const char *host, uint port, bool ipv6)
{
	static sockaddr_in6 host_data;
	
	memset(&host_data, 0, sizeof(host_data));

	struct hostent *h;

	if (!(h = gethostbyname(host)))
		return 0;

	if (ipv6) {
		sockaddr_in6 *p = (sockaddr_in6*) &host_data;
		
		p->sin6_family = AF_INET6;
		p->sin6_port = htons(port);
		memcpy(&p->sin6_addr, h->h_addr, sizeof(p->sin6_addr));
	} else {
		sockaddr_in *p = (sockaddr_in*) &host_data;
		
		p->sin_family = AF_INET;
		p->sin_port = htons(port);
		memcpy(&p->sin_addr, h->h_addr, sizeof(p->sin_addr));
		memset(&p->sin_zero, 0, 8);
	}
	return (&host_data);
}

char *CServiceAttack::get_next_line(FILE * hFile)
{
	static char buffer[1024]; // max line size asigned statically
	uint pos(0);
	char c;

	buffer[pos] = 0;
	while (EOF != (c = getc(hFile)) && pos < 1024) {
		if (c == '\n') {
			if (buffer[0]) {
				buffer[pos] = 0;
				return (buffer);
			}

			pos = 0;
			buffer[pos] = 0;
			continue;
		}
		if (c != ' ') {
			buffer[pos] = c;
			++pos;
		}
	}
	return 0;
}

int CServiceAttack::sendAll(SOCKET sock, bool ssl, void *buffer, size_t n, 
	int flags, long timeout, bool any)
{
	int m_BytesSent;

#ifdef OPENSSL
	if (ssl) {
		m_BytesSent = SSL_write((SSL *) sock, buffer, n);
		return m_BytesSent;
	}
#endif
	
	timeval *ptv = 0;
	timeval tv;
	tv.tv_sec = timeout;
	tv.tv_usec = 0;
	
	if (timeout) 
		ptv = &tv;

	fd_set writefds;
	uint m_BytesTotal(n);
	do {
		FD_ZERO(&writefds);
		FD_SET(sock, &writefds);
		select(sock + 1, 0, &writefds, 0, ptv);
		if (!FD_ISSET(sock, &writefds))
			return ( m_BytesTotal - n );

		m_BytesSent = send(sock, buffer, n, flags);
		if (m_BytesSent <= 0)
			return ( m_BytesTotal - n );
		
		(char&)buffer += m_BytesSent;
		n -= m_BytesSent;
		
		// means we wanted any amount of data
		if (any)
			return ( m_BytesTotal - n );

	} while (n);
	return ( m_BytesTotal - n );
}

int CServiceAttack::recvAll(SOCKET sock, bool ssl, void *buffer, size_t n, int flags,
			    long timeout, bool any)
{
	int m_BytesRead(0);

#ifdef OPENSSL
	if (ssl) {
		m_BytesRead = SSL_read((SSL *) sock, buffer, n);
		return m_BytesRead;
	}
#endif

	timeval *ptv = 0;
	timeval tv;
	tv.tv_sec = timeout;
	tv.tv_usec = 0;
	
	if (timeout) 
		ptv = &tv;

	fd_set readfds;
	uint m_BytesTotal(n);
	do {
		FD_ZERO(&readfds);
		FD_SET(sock, &readfds);
		select(sock + 1, &readfds, 0, 0, ptv);
		if (!FD_ISSET(sock, &readfds))
			return ( m_BytesTotal - n );

		m_BytesRead = recv(sock, buffer, n, flags);
		if (m_BytesRead <= 0)
			return ( m_BytesTotal - n );
		
		(char&)buffer += m_BytesRead;
		n -= m_BytesRead;
		
		// means we wanted any amount of data
		if (any)
			return ( m_BytesTotal - n );

	} while (n);
	return ( m_BytesTotal - n );
}

/*
	revision 1.1.2 *decreted*

void CServiceAttack::rewindBruteForce()
{
	brute_passwd->rewindPasswds(options.min_chars, options.max_chars);
}
*/

SOCKET CServiceAttack::init_connection(thread_data *pData)
{
	SOCKET sock = create_connect(&pData->addr, pData->ssl, (pData->ipv6 ? AF_INET6 : AF_INET));
	while (!sock && pData->state == WORKING) {
		++pData->connection_errors;
		//perror("connect");
		output(MSG_ERROR, "***unable to establish connection\n");
		if (max_connection_errors && pData->connection_errors >= max_connection_errors) {
			output(MSG_ERROR, "***max connection errors reached\n");
			if (pData->state == WORKING)
				pData->state = DONE;
			continue;
		}
		sock = create_connect(&pData->addr, pData->ssl, (pData->ipv6 ? AF_INET6 : AF_INET));
	}
	if (sock)
		pData->connection_errors = 0;
	return sock;
}

CServiceAttack::target_input_data::target_input_data() : type(SERVICE_NONE), 
passwd_options(PASSWD_OPTIONS_NONE), ipv6(false), ssl(false), min_chars(0), max_chars(0)
{
	user = 0;
	charset = 0;
	misc = 0;
	misc_size = 0;
}

CServiceAttack::target_input_data::~target_input_data()
{
	
}

CServiceAttack::target_output_data::target_output_data(thread_data &c, const char *p) :
type(c.type), addr(c.addr), ipv6(c.ipv6), ssl(c.ssl), misc_size(c.misc_size)
{
	user = alloc_copy(c.user);
	passwd = alloc_copy(p);
	if (c.misc_size){
		misc = new char[c.misc_size];
		memcpy(misc, c.misc, c.misc_size);
	}
}

CServiceAttack::target_output_data::target_output_data(target_output_data &c) :
type(c.type), addr(c.addr), ipv6(c.ipv6), ssl(c.ssl), misc_size(c.misc_size)
{
	user = alloc_copy(c.user);
	passwd = alloc_copy(c.passwd);
	if (c.misc_size){
		misc = new char[c.misc_size];
		memcpy(misc, c.misc, c.misc_size);
	}
}

CServiceAttack::target_output_data::~target_output_data()
{
	if (user)
		delete [] user;
	if (passwd)
		delete [] passwd;
	if (misc)
		delete [] misc;
}

CServiceAttack::thread_data::thread_data(target_input_data& d) : type(d.type), 
passwd_options(d.passwd_options), passwd_state(PASSWD_STATE_START), 
min_chars(d.min_chars), max_chars(d.max_chars), addr(d.addr), ipv6(d.ipv6), ssl(d.ssl), 
state(WORKING), connection_errors(0), misc_size(d.misc_size)
{
	user = alloc_copy(d.user);
	charset = alloc_copy(d.charset);
	if (d.misc_size) {
		misc = new char[d.misc_size];
		memcpy(misc, d.misc, d.misc_size);
	}
	
	brute_passwd = 0;
}

CServiceAttack::thread_data::thread_data() : type(SERVICE_NONE), passwd_options(PASSWD_OPTIONS_NONE),
passwd_state(PASSWD_STATE_START), ipv6(false), state(WORKING), connection_errors(0)
{
	user = 0;
	brute_passwd = 0;
	charset = 0;
	misc = 0;
	misc_size = 0;
}

CServiceAttack::thread_data::thread_data(thread_data &c) : type(c.type), 
passwd_options(c.passwd_options), passwd_state(c.passwd_state), addr(c.addr), 
ipv6(c.ipv6), ssl(c.ssl), state(c.state), connection_errors(c.connection_errors)
{
	
/* 
	revision 1.1.1.1

	function decrepted
	 *avoid*
*/
	user = alloc_copy(c.user);
}



CServiceAttack::thread_data::~thread_data() 
{
	if (user)
		delete [] user;
	if (charset)
		delete [] charset;
	if (brute_passwd)
		delete brute_passwd;
	if (misc)
		delete [] misc;
}
