/***************************************************************************
 *  CServiceAttack.h : This file is part of 'chimera'
 *
 *  (c) 2003,2004 by Lukasz Tomicki <tomicki@o2.pl>
 *	Documentation @ http://tomicki.net/
 *
 ****************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef __CSERVICE_ATTACK_H_
#define __CSERVICE_ATTACK_H_

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>

#ifdef LINUX
	#include <sys/socket.h>
	#include <netinet/in.h>
	#include <arpa/inet.h>
	#include <netdb.h>
	#include <fcntl.h>
	#include <unistd.h>
#endif

#ifdef WINDOWS
	#include <windows.h> /* maybe change to MFC later */
	#include <winsock2.h>
#endif

#include <string>
#include <list>
#include <iostream>
#include <fstream>
#include <pthread.h>

#ifdef OPENSSL
	#include <openssl/ssl.h>
	#include <openssl/des.h>
	#include <openssl/err.h>
#endif

#ifdef LIBSSH
	#include <libssh/libssh.h>
#endif

#ifdef WINDOWS
	#define close(x) closesocket(x) /* Windows has to use closesocket(); */
#endif

#include "CBrutePasswd.h"
#include "misc.h"

/* memory manager */
/* use for debug only ! */

#include "new_on.h"
#include "MemoryManager.h"

/* memory manager */

typedef unsigned short int u16;
typedef unsigned int uint;
typedef uint u32;
typedef unsigned char byte;
typedef byte u8;
typedef int SOCKET;

class CServiceAttack {
public:
	typedef enum {
		SERVICE_NONE = 0,
		SERVICE_POP3,
		SERVICE_FTP,
		SERVICE_GG,
		SERVICE_VNC,
		SERVICE_MYSQL,
		SERVICE_HTTP,
		SERVICE_HTTP_PROXY,
		SERVICE_IMAP,
		SERVICE_TELNET,
		SERVICE_SSH,
		SERVICE_NNTP,
		SERVICE_LDAP
	} service_type;
	
	typedef enum {
		PASSWD_OPTIONS_NONE = 0,
		PASSWD_OPTIONS_NULL = 1,
		PASSWD_OPTIONS_LOGIN = 2,
		PASSWD_OPTIONS_LOGIN_REV = 4
	} PasswdOptions;
	
	typedef enum {
		PASSWD_STATE_START,
		PASSWD_STATE_TRY_NULL,
		PASSWD_STATE_TRY_LOGIN,
		PASSWD_STATE_TRY_LOGIN_REV,
		PASSWD_STATE_EXTERN,
		PASSWD_STATE_INTERN,
		PASSWD_STATE_END
	} PasswdStates;
	
	typedef enum {
		MSG_NONE = 0,
		MSG_BASIC,
		MSG_WARNING,
		MSG_ERROR,
		MSG_VERBOSE,
		MSG_DEBUG,
		MSG_ALL
	} MessageLevel;
	
	static const int msg_min = MSG_NONE;
	static const int msg_max = MSG_ALL;	

	class target_input_data {
	public:
		target_input_data();
		~target_input_data();
	
		service_type type;
		const char *user;
		PasswdOptions passwd_options;
		sockaddr_in6 addr;
		bool ipv6;
		bool ssl;
		const char *charset;
		uint min_chars;
		uint max_chars;
		u32 misc_size;
		char *misc;
	};
		
private:
	typedef enum {
		WORKING = 0,
		DONE,
		FINISHED
	} thread_data_state;
	
	class thread_data {
	public:
		thread_data();
		thread_data(thread_data&);
		thread_data(target_input_data&);
		~thread_data();
	
		service_type type;
		char *user;
		byte passwd_options;
		byte passwd_state;
		uint min_chars;
		uint max_chars;
		sockaddr_in6 addr;
		bool ipv6;
		bool ssl;
		thread_data_state state;
		uint connection_errors;
		CBrutePasswd *brute_passwd;
		const char *charset;
		u32 misc_size;
		char *misc;
	};
	
	typedef std::list < thread_data * >::iterator cracking_dataIter;

public:
	class target_output_data {
	public:
		target_output_data(thread_data&, const char*);
		target_output_data(target_output_data&);
		~target_output_data();
	
		service_type type;
		const char* user;
		const char* passwd;
		sockaddr_in6 addr;
		bool ipv6;
		bool ssl;
		u32 misc_size;
		char *misc;
	};	

public:
	static const uint autobalance = 64;
	
public:
	CServiceAttack();
	~CServiceAttack();

	void add_target(target_input_data &start_options, bool = false);
	void run_atack();
	void set_output_level(MessageLevel level);
	MessageLevel get_output_level();
	void set_threads(uint n);
	uint get_threads();
	uint get_thread_creation_speed();
	void set_thread_creation_speed(uint);
	bool get_thread_reduction();
	void set_thread_reduction(bool);
	void set_output(void (*p)(const char*));
	void set_passwd_func(const char* (*p)(bool));
	void set_notice(void (*p)(target_output_data*));
	void set_conn_timeout(uint);
	uint get_conn_timeout();
	void set_conn_errors_max(uint);
	uint get_conn_errors_max();

	bool is_working();
	void flush();

	// static support functions
	static sockaddr_in6 *get_host_data(const char *host, uint, bool);
	static void *service_wrapper(void *obj);
	static char *get_next_line(FILE * hFile);
	static const char* rev_string(const char *p);

protected:
	
	// service attack functions 
	int service_pop3(thread_data *pData);
	int service_ftp(thread_data *pData);
	int service_vnc(thread_data *pData);
	int service_nntp(thread_data *pData);
	int service_ssh(thread_data *pData);
	int service_imap(thread_data *pData);
	int service_gg(thread_data *pData);
	int service_ldap(thread_data *pData);
	int service_telnet(thread_data *pData);

	/* helper telnet functions */
	/*       *start*           */

	void make_lowercase(char* p);

	/* helper telnet functions */
	/*       *end*           */

	int service_http(thread_data *pData);
	
	/*
		for service http
	 	*misc = "[4 bytes : length of request]
		[x byte request (format: (http | https)://servername/remote_path/'\0')]
		[4 bytes : length of remote server name]
		[y byte remote server (format: remote_server'\0')]
		
		misc_size = 4 + x + 4 + y;
	*/
	
	int service_http_proxy(thread_data *pData);
	
	/*
		for service http-proxy
	 	*misc = "[4 bytes : length of request]
		[x byte proxy request (format: (http | https)://servername/remote_path/'\0')]
		[4 bytes : length of remote server name]
		[y byte remote server (format: remote_server'\0')]
		
		misc_size = 4 + x + 4 + y;
	*/
	
	/* helper http, http-proxy functions */
	/*       *start*                     */
	
	void convert_to_base64(u8*);
	u8 convert_64(u8);

	/* helper http, http-proxy functions */
	/*       *end*                       */
	
	int service_mysql(thread_data *pData);
	
	/* helper mysql structures */
	/*       *start*           */

	struct mysql_rand_struct {
		unsigned long seed1, seed2, max_value;
	  	double max_value_dbl;
	};
	
	/* helper mysql structures */
	/*       *end*             */
	
	/* helper mysql functions */
	/*       *start*          */
	
	bool mysql_init(int, char*, thread_data *pData);
	const char *mysql_scramble(char*, const char*, const char*);
	void mysql_hash_password(unsigned long*, const char*);
	double mysql_rnd(mysql_rand_struct*);
	void mysql_randominit(mysql_rand_struct*, unsigned long, unsigned long);
	void mysql_send_com_quit(int, thread_data*);
	bool mysql_parse_response(u8*);
	char* mysql_prepare_auth(const char*, const char*, char*);

	
	/* helper mysql functions */
	/*       *end*            */
	
	// other helper functions
	
	void adjust_connections(thread_data *pData);
	void service_atack();
	void account_cracked(thread_data*, const char*);
	SOCKET create_connect(sockaddr_in6 * addr, bool ssl, int = AF_INET, int =
			      SOCK_STREAM, int = 0);
	SOCKET init_connection(thread_data *pData);
	void output(MessageLevel level, char *text, ...);
	const char* get_next_passwd(thread_data*, char*, u32);
	int sendAll(SOCKET socket, bool ssl, void *buffer, size_t n, int flags = 0, 
		long = 60, bool = true);
	int recvAll(SOCKET socket, bool ssl, void *buffer, size_t n, int flags = 0, 
		long = 60, bool = true);

#ifdef OPENSSL
	static RSA *ssl_temp_rsa_cb(SSL * ssl, int, int keylength);
#endif

private:
	pthread_cond_t thread_wait;
	void (*write_output)(const char*);
	const char* (*next_passwd)(bool);
	void (*notice_func)(target_output_data*);
	std::list < thread_data* >cracking_data;
	std::list < thread_data* >old_data;
	uint threads_running;
	uint threads;
	uint output_level;
	bool thread_reduce;
	uint thread_creation_speed;
	uint std_timeout;
	uint max_connection_errors;

#ifdef OPENSSL
	static RSA *rsa;
#endif
};

#ifndef __BRUTE_PASSWD_H_

template < class tName > tName min(tName & a, tName & b)
{
	return (a > b ? b : a);
}

template < class tName > tName max(tName & a, tName & b)
{
	return (a > b ? a : b);
}

#endif // __BRUTE_PASSWD_H_

#endif // __CSERVICE_ATTACK_H_
